﻿<?php

$CONFIG_FILE = __DIR__ . '/../../config.php';
if (is_file($CONFIG_FILE)) {
    require_once $CONFIG_FILE;
}

if (!defined('NCI_DEBUG')) { define('NCI_DEBUG', false); }


if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

if (!function_exists('nci_flash_set')) {
    function nci_flash_set(string $key, $value): void {
        $_SESSION[$key] = $value;
    }
}
if (!function_exists('nci_flash_take')) {
    function nci_flash_take(string $key, $default = null) {
        if (!isset($_SESSION[$key])) return $default;
        $v = $_SESSION[$key];
        unset($_SESSION[$key]);
        return $v;
    }
}

if (!function_exists('nci_build_pdo_from_config')) {
    function nci_build_pdo_from_config(): PDO {
        if (defined('DB_DSN')) {
            $dsn  = DB_DSN;
            $user = defined('DB_USER') ? DB_USER : null;
            $pass = defined('DB_PASS') ? DB_PASS : null;
            $opts = defined('DB_OPTIONS') && is_array(DB_OPTIONS) ? DB_OPTIONS : [];
            return new PDO($dsn, $user, $pass, $opts + [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
        }
        if (isset($GLOBALS['config']) && is_array($GLOBALS['config']) && isset($GLOBALS['config']['db'])) {
            $db = $GLOBALS['config']['db'];
            if (!empty($db['dsn'])) {
                $dsn = $db['dsn'];
            } else {
                $host    = $db['host']    ?? '127.0.0.1';
                $name    = $db['name']    ?? 'nova';
                $charset = $db['charset'] ?? 'utf8mb4';
                $dsn = "mysql:host={$host};dbname={$name};charset={$charset}";
            }
            $user = $db['user'] ?? 'root';
            $pass = $db['pass'] ?? '';
            $opts = is_array($db['options'] ?? null) ? $db['options'] : [];
            return new PDO($dsn, $user, $pass, $opts + [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
        }
        $host    = $GLOBALS['DB_HOST']    ?? '127.0.0.1';
        $name    = $GLOBALS['DB_NAME']    ?? 'nova';
        $user    = $GLOBALS['DB_USER']    ?? 'root';
        $pass    = $GLOBALS['DB_PASS']    ?? '';
        $charset = $GLOBALS['DB_CHARSET'] ?? 'utf8mb4';
        $dsn = "mysql:host={$host};dbname={$name};charset={$charset}";
        return new PDO($dsn, $user, $pass, [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
    }
}

if (!function_exists('client_ip')) {
    function client_ip(): string {
        $keys = ['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','HTTP_CLIENT_IP','REMOTE_ADDR'];
        foreach ($keys as $k) if (!empty($_SERVER[$k])) return explode(',', $_SERVER[$k])[0];
        return '0.0.0.0';
    }
}
if (!function_exists('nci_only_digits')) {
    function nci_only_digits(string $s): string { return preg_replace('/\D+/', '', $s); }
}
if (!function_exists('nci_h')) {
    function nci_h(?string $s): string { return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8'); }
}

$pdo = null;
$errors = [];
$affiliate = null;

$ref = isset($_GET['ref']) ? trim((string)$_GET['ref']) : '';

try {
    $pdo = nci_build_pdo_from_config();
    if ($ref !== '') {
        $stmt = $pdo->prepare("SELECT id, name, code FROM affiliates WHERE code = ? AND is_active = 1 LIMIT 1");
        $stmt->execute([$ref]);
        $affiliate = $stmt->fetch() ?: null;
    }
} catch (Throwable $e) {
    if (NCI_DEBUG) error_log("DB prefetch error: " . $e->getMessage());
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email    = trim($_POST['user_email'] ?? '');
    $phone    = trim($_POST['user_phone'] ?? '');
    $locale   = trim($_POST['preferred_locale'] ?? 'en');
    $pass     = (string)($_POST['user_password'] ?? '');
    $cpass    = (string)($_POST['confirm_password'] ?? '');
    $terms    = isset($_POST['terms']) ? (string)$_POST['terms'] : '';
    $ref_post = trim($_POST['ref_code'] ?? '');

    $errors = [];

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) { $errors[] = "Please enter a valid email address."; }
    if ($phone === '') { $errors[] = "Phone number is required."; }
    if (strlen($pass) < 8) { $errors[] = "Password must be at least 8 characters."; }
    if (strlen($cpass) < 8) { $errors[] = "Confirm Password must be at least 8 characters."; }
    if ($pass !== $cpass) { $errors[] = "Passwords do not match."; }
    if ($terms !== '1')   { $errors[] = "You must accept the Terms & Conditions."; }

    $phone_digits = nci_only_digits($phone);
    if ($phone !== '' && $phone_digits === '') { $errors[] = "Please enter a valid phone number."; }

    $resolvedAffiliate = null;
    if ($ref_post !== '' && $pdo) {
        try {
            $stmt = $pdo->prepare("SELECT id, name, code FROM affiliates WHERE code = ? AND is_active = 1 LIMIT 1");
            $stmt->execute([$ref_post]);
            $resolvedAffiliate = $stmt->fetch() ?: null;
        } catch (Throwable $e) {
            if (NCI_DEBUG) error_log("Affiliate lookup error: " . $e->getMessage());
        }
    }

    if ($pdo && !$errors) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
            $stmt->execute([$email]);
            if ($stmt->fetch()) $errors[] = "This email is already registered.";

            if ($phone_digits !== '') {
                $stmt = $pdo->prepare("SELECT id FROM users WHERE phone_digits = ? LIMIT 1");
                $stmt->execute([$phone_digits]);
                if ($stmt->fetch()) $errors[] = "This phone number is already registered.";
            }
        } catch (Throwable $e) {
            if (NCI_DEBUG) error_log("Uniqueness check error: " . $e->Message());
            $errors[] = "Temporary database issue. Please try again shortly.";
        }
    } elseif (!$pdo) {
        $errors[] = "Temporary database issue. Please try again shortly.";
    }

    if ($errors) {
        nci_flash_set('signup_errors', $errors);
        nci_flash_set('signup_old', [
            'user_email'       => $email,
            'user_phone'       => $phone,
            'preferred_locale' => $locale,
            'ref_code'         => $ref_post,
        ]);
        $redir = '/novademo/html/demo1/index.php';
        if ($ref_post !== '') {
            $redir .= '?ref=' . rawurlencode($ref_post);
        }
        $redir .= '#register';
        header('Location: ' . $redir);
        exit;
    }

    try {
        $password_hash = password_hash($pass, PASSWORD_BCRYPT);
        $now_ip = client_ip();

        if ($resolvedAffiliate) {
            $affiliate_id        = (int)$resolvedAffiliate['id'];
            $affiliate_code_val  = (string)$resolvedAffiliate['code'];
            $user_source_type    = "afiliate";             // per spec
            $user_source_label   = (string)$resolvedAffiliate['name'];
        } else {
            $affiliate_id        = null;
            $affiliate_code_val  = null;
            $user_source_type    = "website";
            $user_source_label   = "from website";
        }

        $sql = "INSERT INTO users
            (email, phone, phone_digits, is_active, password_hash, registration_channel,
             preferred_locale, affiliate_id, affiliate_code_value,
             user_source_type, user_source_label,
             current_status_id, current_status_set_at, current_assignment_id,
             current_desk_id, current_manager_user_id, current_agent_user_id,
             last_login_at, last_login_ip, created_ip, updated_ip, created_at, updated_at)
            VALUES
            (:email, :phone, :phone_digits, :is_active, :password_hash, :registration_channel,
             :preferred_locale, :affiliate_id, :affiliate_code_value,
             :user_source_type, :user_source_label,
             :current_status_id, :current_status_set_at, :current_assignment_id,
             :current_desk_id, :current_manager_user_id, :current_agent_user_id,
             :last_login_at, :last_login_ip, :created_ip, :updated_ip, NOW(), NOW())";

        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':email',                $email);
        $stmt->bindValue(':phone',                $phone, PDO::PARAM_STR);
        $stmt->bindValue(':phone_digits',         $phone_digits, PDO::PARAM_STR);
        $stmt->bindValue(':is_active',            1, PDO::PARAM_INT);
        $stmt->bindValue(':password_hash',        $password_hash);
        $stmt->bindValue(':registration_channel', 'website');
        $stmt->bindValue(':preferred_locale',     $locale !== '' ? $locale : 'en');
        if ($affiliate_id === null) {
            $stmt->bindValue(':affiliate_id', null, PDO::PARAM_NULL);
            $stmt->bindValue(':affiliate_code_value', null, PDO::PARAM_NULL);
        } else {
            $stmt->bindValue(':affiliate_id', $affiliate_id, PDO::PARAM_INT);
            $stmt->bindValue(':affiliate_code_value', $affiliate_code_val, PDO::PARAM_STR);
        }
        $stmt->bindValue(':user_source_type',     $user_source_type);
        $stmt->bindValue(':user_source_label',    $user_source_label);
        $stmt->bindValue(':current_status_id',        null, PDO::PARAM_NULL);
        $stmt->bindValue(':current_status_set_at',    null, PDO::PARAM_NULL);
        $stmt->bindValue(':current_assignment_id',    null, PDO::PARAM_NULL);
        $stmt->bindValue(':current_desk_id',          null, PDO::PARAM_NULL);
        $stmt->bindValue(':current_manager_user_id',  null, PDO::PARAM_NULL);
        $stmt->bindValue(':current_agent_user_id',    null, PDO::PARAM_NULL);
        $stmt->bindValue(':last_login_at',            null, PDO::PARAM_NULL);
        $stmt->bindValue(':last_login_ip',            null, PDO::PARAM_NULL);
        $stmt->bindValue(':created_ip',               $now_ip);
        $stmt->bindValue(':updated_ip',               $now_ip);

        $stmt->execute();

        header("Location: /novademo/html/demo1/index.php#login");
        exit;
    } catch (Throwable $e) {
        if (NCI_DEBUG) error_log("INSERT error: " . $e->getMessage());
        nci_flash_set('signup_errors', ["Could not create your account. Please try again."]);
        nci_flash_set('signup_old', [
            'user_email'       => $email,
            'user_phone'       => $phone,
            'preferred_locale' => $locale,
            'ref_code'         => $ref_post,
        ]);
        $redir = '/novademo/html/demo1/index.php';
        if ($ref_post !== '') $redir .= '?ref=' . rawurlencode($ref_post);
        $redir .= '#register';
        header('Location: ' . $redir);
        exit;
    }
}

?>
<!DOCTYPE html>
<html class="h-full" data-kt-theme="true" data-kt-theme-mode="light" dir="ltr" lang="en">
<head>
  <base href="../../../../">
  <title>Nova Chain Investments - Sign Up</title>
  <meta charset="utf-8"/>
  <meta name="robots" content="follow, index"/>
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no"/>
  <meta name="description" content="Sign up for Nova Chain Investments"/>
  <link rel="apple-touch-icon" sizes="180x180" href="assets/media/app/apple-touch-icon.png"/>
  <link rel="icon" type="image/png" sizes="32x32" href="assets/media/app/favicon-32x32.png"/>
  <link rel="icon" type="image/png" sizes="16x16" href="assets/media/app/favicon-16x16.png"/>
  <link rel="shortcut icon" href="assets/media/app/favicon.ico"/>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet"/>
  <link href="assets/vendors/apexcharts/apexcharts.css" rel="stylesheet"/>
  <link href="assets/vendors/keenicons/styles.bundle.css" rel="stylesheet"/>
  <link href="assets/css/styles.css" rel="stylesheet"/>
</head>
<body class="antialiased flex h-full text-base text-foreground bg-background">
  <!-- Theme Mode -->
  <script>
    const defaultThemeMode='light';let themeMode;
    if(document.documentElement){
      if(localStorage.getItem('kt-theme')){themeMode=localStorage.getItem('kt-theme');}
      else if(document.documentElement.hasAttribute('data-kt-theme-mode')){themeMode=document.documentElement.getAttribute('data-kt-theme-mode');}
      else{themeMode=defaultThemeMode;}
      if(themeMode==='system'){themeMode=window.matchMedia('(prefers-color-scheme: dark)').matches?'dark':'light';}
      document.documentElement.classList.add(themeMode);
    }
  </script>

  <!-- Page -->
  <main id="page-content" class="flex items-center justify-center grow min-h-screen bg-background">
    <div class="kt-card max-w-[370px] w-full">
      <form id="sign_up_form" class="kt-card-content flex flex-col gap-5 p-10"
            method="post"
            action="/novademo/html/demo1/authentication/classic/sign-up.php"
            accept-charset="UTF-8">
        <!-- Header -->
        <div class="text-center mb-2.5">
          <h3 class="text-lg font-medium text-mono leading-none mb-2.5">Sign Up</h3>
          <div class="flex items-center justify-center">
            <span class="text-sm text-secondary-foreground me-1.5">Already have an account?</span>
            <a class="text-sm link" href="/novademo/html/demo1/authentication/classic/sign-in.php" data-page="signin">Sign In</a>
          </div>
        </div>

        <?php
          $flashedErrors = nci_flash_take('signup_errors', []);
          $flashedOld    = nci_flash_take('signup_old', []);
          if ($flashedErrors) $errors = $flashedErrors;

          $old_email = $_POST['user_email'] ?? ($flashedOld['user_email'] ?? '');
          $old_phone = $_POST['user_phone'] ?? ($flashedOld['user_phone'] ?? '');
          $old_locale = $_POST['preferred_locale'] ?? ($flashedOld['preferred_locale'] ?? 'en');
          $ref = $_GET['ref'] ?? ($flashedOld['ref_code'] ?? $ref);
        ?>

        <?php if ($errors): ?>
          <div class="rounded-md border border-destructive/40 bg-destructive/10 text-destructive px-3 py-2 text-sm">
            <?php foreach ($errors as $e): ?>
              <div>• <?= nci_h($e) ?></div>
            <?php endforeach; ?>
          </div>
        <?php endif; ?>

        <!-- Email -->
        <div class="flex flex-col gap-1">
          <label class="kt-form-label text-mono" for="user_email">Email</label>
          <input id="user_email" class="kt-input" name="user_email" type="email" placeholder="email@email.com"
                 required value="<?= nci_h($old_email) ?>"/>
        </div>

        <!-- Phone (required) -->
        <div class="flex flex-col gap-1">
          <label class="kt-form-label text-mono" for="user_phone">Phone</label>
          <input id="user_phone" class="kt-input" name="user_phone" type="tel" placeholder="+995 555 123 456"
                 required value="<?= nci_h($old_phone) ?>"/>
        </div>

        <!-- Preferred Language -->
        <div class="flex flex-col gap-1">
          <label class="kt-form-label text-mono" for="preferred_locale">Language</label>
          <select id="preferred_locale" name="preferred_locale" class="kt-input" required>
            <?php
              $langs = [
                  'en' => 'English',
                  'de' => 'Deutsch',
                  'es' => 'Espa&ntilde;ol',
              ];
              foreach ($langs as $k=>$v) {
                  $selected = ($old_locale === $k) ? 'selected' : '';
                  echo '<option value="'.nci_h($k).'" '.$selected.'>'.$v.'</option>';
              }
            ?>
          </select>
        </div>

        <!-- Password -->
        <div class="flex flex-col gap-1">
          <label class="kt-form-label font-normal text-mono" for="user_password">Password</label>
          <div class="kt-input" data-kt-toggle-password="true">
            <input id="user_password" name="user_password" placeholder="Enter Password" type="password"
                   required minlength="8" pattern=".{8,}" />
            <button class="kt-btn kt-btn-sm kt-btn-ghost kt-btn-icon bg-transparent! -me-1.5"
                    data-kt-toggle-password-trigger="true" type="button">
              <span class="kt-toggle-password-active:hidden">
                <i class="ki-filled ki-eye text-muted-foreground"></i>
              </span>
              <span class="hidden kt-toggle-password-active:block">
                <i class="ki-filled ki-eye-slash text-muted-foreground"></i>
              </span>
            </button>
          </div>
          <span class="text-xs text-muted-foreground">At least 8 characters.</span>
        </div>

        <!-- Confirm Password -->
        <div class="flex flex-col gap-1">
          <label class="kt-form-label font-normal text-mono" for="confirm_password">Confirm Password</label>
          <div class="kt-input" data-kt-toggle-password="true">
            <input id="confirm_password" name="confirm_password" placeholder="Re-enter Password" type="password"
                   required minlength="8" pattern=".{8,}" />
            <button class="kt-btn kt-btn-sm kt-btn-ghost kt-btn-icon bg-transparent! -me-1.5"
                    data-kt-toggle-password-trigger="true" type="button">
              <span class="kt-toggle-password-active:hidden">
                <i class="ki-filled ki-eye text-muted-foreground"></i>
              </span>
              <span class="hidden kt-toggle-password-active:block">
                <i class="ki-filled ki-eye-slash text-muted-foreground"></i>
              </span>
            </button>
          </div>
        </div>

        <!-- Terms -->
        <label class="kt-checkbox-group">
          <input class="kt-checkbox kt-checkbox-sm" name="terms" type="checkbox" value="1" required/>
          <span class="kt-checkbox-label">
            I accept <a class="text-sm link" href="#">Terms &amp; Conditions</a>
          </span>
        </label>

        <!-- Hidden ref (mirrors ?ref=) -->
        <input type="hidden" name="ref_code" value="<?= nci_h($ref) ?>"/>

        <?php if ($affiliate): ?>
          <div class="text-xs text-secondary-foreground">
            Referred by: <strong><?= nci_h($affiliate['name']) ?></strong>
          </div>
        <?php elseif ($ref !== ''): ?>
          <div class="text-xs text-destructive">
            The referral code “<?= nci_h($ref) ?>” is invalid or inactive; you can still sign up.
          </div>
        <?php endif; ?>

        <!-- Submit -->
        <button
          type="submit"
          class="kt-btn kt-btn-primary"
          style="width:max-content!important; display:inline-flex!important; align-self:flex-start!important; justify-content:center; gap:.5rem; padding:.6rem 1rem;">
          Sign Up
        </button>
      </form>
    </div>
  </main>

  <!-- Scripts -->
  <script src="assets/js/core.bundle.js"></script>
  <script src="assets/vendors/ktui/ktui.min.js"></script>
  <script src="assets/vendors/apexcharts/apexcharts.min.js"></script>

  <!-- Client-side password + confirm validation -->
  <script>
    (function(){
      const form  = document.getElementById('sign_up_form');
      const pass1 = document.getElementById('user_password');
      const pass2 = document.getElementById('confirm_password');

      function validatePasswords(){
        pass1.setCustomValidity(pass1.value.length < 8 ? 'Password must be at least 8 characters.' : '');
        pass2.setCustomValidity(pass2.value.length < 8 ? 'Confirm Password must be at least 8 characters.' : '');
        if (pass1.value.length >= 8 && pass2.value.length >= 8 && pass1.value !== pass2.value) {
          pass2.setCustomValidity('Passwords do not match.');
        }
        if (pass1.value === pass2.value && pass1.value.length >= 8 && pass2.value.length >= 8) {
          pass2.setCustomValidity('');
        }
      }

      pass1.addEventListener('input', validatePasswords);
      pass2.addEventListener('input', validatePasswords);

      form.addEventListener('submit', function(e){
        validatePasswords();
        if (!form.checkValidity()) {
          e.preventDefault();
          e.stopPropagation();
        }
      });
    })();
  </script>
</body>
</html>
