<?php
// /novademo/html/demo1/send_contact.php
declare(strict_types=1);


const ADMIN_EMAIL = 'admin@test.com';

// Central URLs
$baseUrl    = '/novademo/html/demo1';
$contactUrl = $baseUrl . '/contact.php';
$homeUrl    = $baseUrl . '/index.php';


$autoloadPath = __DIR__ . '/vendor/autoload.php';
if (!file_exists($autoloadPath)) {
  renderPage(
    'Setup Required',
    'PHPMailer is not installed. Run <code>composer require phpmailer/phpmailer</code> in <code>' . htmlspecialchars(__DIR__) . '</code>.',
    $homeUrl,
    $contactUrl
  );
  exit;
}
require $autoloadPath;

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;


function clean(string $s): string {
  return trim(str_replace(["\r", "\n"], ' ', $s));
}
function client_ip(): string {
  foreach (['HTTP_CLIENT_IP','HTTP_X_FORWARDED_FOR','REMOTE_ADDR'] as $k) {
    if (!empty($_SERVER[$k])) return (string)$_SERVER[$k];
  }
  return '0.0.0.0';
}


function renderPage(string $title, string $messageHtml, string $homeUrl, string $contactUrl, bool $ok = false): void {
  $statusClass = $ok ? 'text-emerald-600' : 'text-destructive-foreground';
  echo '<!DOCTYPE html>
<html class="h-full" data-kt-theme="true" data-kt-theme-mode="light" lang="en" dir="ltr">
<head>
  <base href="../../">
  <meta charset="utf-8"/>
  <title>' . htmlspecialchars($title) . '</title>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet"/>
  <link href="assets/vendors/keenicons/styles.bundle.css" rel="stylesheet"/>
  <link href="assets/css/styles.css" rel="stylesheet"/>
  <!-- Keep a minimal head; the SPA will not carry head styles. -->
</head>
<body class="antialiased flex h-full text-base text-foreground bg-background">
  <main id="page-content" class="w-full flex justify-center py-12">
    <!-- LOCAL PATCH so text stays visible when injected via SPA -->
    <style>
      /* Scope to this payload only */
      #page-content a.kt-btn{position:relative;display:inline-flex;align-items:center;justify-content:center;overflow:visible;isolation:isolate}
      #page-content a.kt-btn .btn-label,
      #page-content a.kt-btn > span,
      #page-content a.kt-btn > *{position:relative;z-index:2!important;pointer-events:none}
      #page-content a.kt-btn::before,#page-content a.kt-btn::after{z-index:1!important}
      #page-content a.kt-btn.kt-btn-primary{color:#fff!important}
    </style>

    <div class="kt-card max-w-[370px] w-full">
      <div class="kt-card-content flex flex-col gap-5 p-10">
        <div class="text-center">
          <h3 class="text-lg font-medium text-mono leading-none mb-2.5">' . htmlspecialchars($title) . '</h3>
          <div class="text-sm ' . $statusClass . '">' . $messageHtml . '</div>
        </div>
        <div class="flex gap-2">
          <a class="kt-btn kt-btn-primary grow justify-center" href="' . htmlspecialchars($homeUrl) . '#contact" data-page="contact">
            <span class="btn-label">Back to Contact</span>
          </a>
          <a class="kt-btn kt-btn-ghost grow justify-center" href="' . htmlspecialchars($homeUrl) . '">
            <span class="btn-label">Home</span>
          </a>
        </div>
      </div>
    </div>
  </main>
  <script src="assets/js/core.bundle.js"></script>
  <script src="assets/vendors/ktui/ktui.min.js"></script>
</body>
</html>';
}


if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  renderPage('Invalid Request', 'Only POST requests are allowed.', $homeUrl, $contactUrl, false);
  exit;
}

// ---------- ANTI-BOT ----------
$honeypot = $_POST['website'] ?? '';
if (!empty($honeypot)) {
  renderPage('Message Sent', 'Thank you! If needed, our team will reply shortly.', $homeUrl, $contactUrl, true);
  exit;
}

// ---------- INPUTS ----------
$name    = clean($_POST['name']    ?? '');
$email   = clean($_POST['email']   ?? '');
$message = trim($_POST['message']  ?? '');
$consent = isset($_POST['consent']);

// ---------- VALIDATION ----------
$errors = [];
if ($name === '')   $errors[] = 'Name is required.';
if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'A valid email address is required.';
if ($message === '' || mb_strlen($message) < 5) $errors[] = 'Please enter a message (min 5 characters).';

if ($errors) {
  renderPage(
    'Please fix the following',
    '<ul class="list-disc list-inside">' .
      implode('', array_map(fn($e) => '<li>'.htmlspecialchars($e).'</li>', $errors)) .
    '</ul>',
    $homeUrl,
    $contactUrl,
    false
  );
  exit;
}

// ---------- BUILD EMAIL ----------
$subject  = 'New Contact Form Message';
$bodyHtml = '
  <p><strong>Name:</strong> ' . htmlspecialchars($name) . '</p>
  <p><strong>Email:</strong> ' . htmlspecialchars($email) . '</p>
  <p><strong>Consent to be contacted:</strong> ' . ($consent ? 'Yes' : 'No') . '</p>
  <p><strong>Message:</strong><br>' . nl2br(htmlspecialchars($message)) . '</p>
  <hr>
  <p style="font-size:12px;color:#666">
    IP: ' . htmlspecialchars(client_ip()) . '<br>
    User-Agent: ' . htmlspecialchars($_SERVER['HTTP_USER_AGENT'] ?? 'n/a') . '
  </p>
';
$bodyText = "Name: {$name}\nEmail: {$email}\nConsent: " . ($consent ? 'Yes' : 'No') .
            "\n\nMessage:\n{$message}\n\nIP: " . client_ip();

// ---------- SEND VIA SMTP (PHPMailer) ----------
$mail = new PHPMailer(true);

try {
  // Server settings 
  $mail->isSMTP();
  $mail->Host       = 'smtp.test.local';
  $mail->SMTPAuth   = true;
  $mail->Username   = 'smtp_user@test.local';
  $mail->Password   = 'smtp_password_here';
  $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
  $mail->Port       = 587;

  // From/To
  $mail->setFrom('no-reply@test.com', 'Nova Chain Website');
  $mail->addAddress(ADMIN_EMAIL, 'Site Admin');

  // Let replies go to the user
  $mail->addReplyTo($email, $name);

  // Content
  $mail->isHTML(true);
  $mail->Subject = $subject;
  $mail->Body    = $bodyHtml;
  $mail->AltBody = $bodyText;
  $mail->CharSet = 'UTF-8';

  $mail->send();

  renderPage(
    'Message Sent',
    'Thank you! Your message has been sent successfully. Our team will get back to you shortly.',
    $homeUrl,
    $contactUrl,
    true
  );

} catch (Exception $e) {
  $reason = htmlspecialchars($mail->ErrorInfo ?: $e->getMessage());
  renderPage(
    'Could not send your message',
    'We encountered an issue while sending your message. Please try again later.' .
    '<div class="text-xs text-secondary-foreground mt-3">Details: ' . $reason . '</div>',
    $homeUrl,
    $contactUrl,
    false
  );
}
?>