<?php

if (session_status() === PHP_SESSION_NONE) { session_start(); }

// ---- Load DB config (../config.php) ----
$__cfg = dirname(__DIR__) . '/config.php';
if (!is_file($__cfg)) { die('config.php not found at /novademo/html/demo1/config.php'); }
require_once $__cfg;

$pdo = $pdo ?? null;
if (!$pdo) {
    $dbHost = defined('DB_HOST') ? DB_HOST : null;
    $dbName = defined('DB_NAME') ? DB_NAME : null;
    $dbUser = defined('DB_USER') ? DB_USER : null;
    $dbPass = defined('DB_PASS') ? DB_PASS : null;
    if ($dbHost && $dbName && $dbUser !== null) {
        try {
            $pdo = new PDO(
                "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4",
                $dbUser,
                $dbPass ?? '',
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]
            );
        } catch (Throwable $e) {
            die("DB connection failed: " . htmlspecialchars($e->getMessage()));
        }
    } else {
        die("Database config missing. Provide \$pdo or DB_* constants in config.php.");
    }
}

// ---- Auth ----
$currentUserId = $_SESSION['user_id'] ?? null;
if (!$currentUserId) {
    ?>
    <main id="page-content" class="w-full flex justify-center py-12">
      <div class="kt-card max-w-[600px] w-full">
        <div class="kt-card-content flex flex-col gap-5 p-10">
          <h2 class="text-xl font-semibold text-center">Assets</h2>
          <div class="text-sm p-4 rounded-lg bg-accent/40 text-secondary-foreground text-center">
            You must be logged in to manage assets.
            <a href="#login" data-page="login" class="text-primary underline ml-1">Go to Login</a>
          </div>
        </div>
      </div>
    </main>
    <?php
    return;
}

// ---- CSRF ----
if (empty($_SESSION['csrf_token'])) { $_SESSION['csrf_token'] = bin2hex(random_bytes(16)); }
$csrf_token = $_SESSION['csrf_token'];

$flash = $_SESSION['asset_flash'] ?? ['success'=>[], 'error'=>[]];
unset($_SESSION['asset_flash']);

$old   = $_SESSION['asset_old'] ?? ['asset_id'=>'', 'amount'=>''];
unset($_SESSION['asset_old']);

$editState = $_SESSION['asset_edit'] ?? null; // ['id'=>..., 'asset_id'=>..., 'amount'=>..., 'decimals'=>...]
$formMode  = $editState ? 'update' : 'add';

// ---- Helper fns ----
function fetchActiveAssetsForDropdown(PDO $pdo): array {
    return $pdo->query("SELECT id, symbol, name, decimals
                          FROM assets
                         WHERE is_active = 1
                      ORDER BY symbol ASC, name ASC")->fetchAll();
}
function fetchClientAssets(PDO $pdo, int $uid, int $isActive): array {
    $st = $pdo->prepare("SELECT ca.id, ca.asset_id, ca.amount, ca.is_active, ca.updated_at,
                                a.symbol, a.name, a.decimals
                           FROM client_assets ca
                           JOIN assets a ON a.id = ca.asset_id
                          WHERE ca.client_user_id = :u AND ca.is_active = :ia
                       ORDER BY a.symbol, a.name, ca.id");
    $st->execute([':u'=>$uid, ':ia'=>$isActive]);
    return $st->fetchAll();
}
function decimalsToStepAttr($dec): string {
    if ($dec === null || $dec === '' || !is_numeric($dec)) return 'any';
    $dec = (int)$dec;
    if ($dec <= 0) return '1';
    return '0.' . str_repeat('0', max(0, $dec-1)) . '1';
}

// ---- Data for dropdown and lists ----
$dropdownAssets = fetchActiveAssetsForDropdown($pdo);
$clientActive   = fetchClientAssets($pdo, $currentUserId, 1);
$clientInactive = fetchClientAssets($pdo, $currentUserId, 0);

// ---- Prefill logic ----
$prefill = ['asset_id' => '', 'amount' => '', 'decimals' => null];
if ($formMode === 'update' && $editState) {
    $prefill['asset_id'] = (string)$editState['asset_id'];
    $prefill['amount']   = rtrim(rtrim((string)$editState['amount'], '0'), '.');
    $prefill['decimals'] = isset($editState['decimals']) ? (int)$editState['decimals'] : null;
} else {
    $prefill['asset_id'] = (string)$old['asset_id'];
    $prefill['amount']   = (string)$old['amount'];
    if ($prefill['asset_id'] !== '') {
        foreach ($dropdownAssets as $a) {
            if ((int)$a['id'] === (int)$prefill['asset_id']) { $prefill['decimals'] = (int)$a['decimals']; break; }
        }
    }
}

// ---- URLs ----
$BASE_URL    = '/novademo/html/demo1';
$HANDLER_URL = $BASE_URL . '/userdashboard/assets_handler.php';
$RETURN_TO   = $BASE_URL . '/index.php#assets';
?>

<!-- Assets Section -->
<section id="assets" class="py-8" style="width:70%;max-width:1100px;margin:0 auto;">
  <div class="kt-card">
    <div class="kt-card-content p-10 flex flex-col gap-5">
      <div class="text-center">
        <h3 class="text-lg font-medium text-mono">Manage Your Assets</h3>
        <span class="text-sm text-secondary-foreground">Request new assets or update existing ones</span>
      </div>

      <!-- Flash messages -->
      <?php foreach (($flash['success'] ?? []) as $m): ?>
        <div class="kt-form-label text-green-600" role="status"><?= htmlspecialchars($m) ?></div>
      <?php endforeach; ?>
      <?php foreach (($flash['error'] ?? []) as $m): ?>
        <div class="kt-form-label text-red-600" role="alert"><?= htmlspecialchars($m) ?></div>
      <?php endforeach; ?>

      <!-- Request / Update Form (posts to handler) -->
      <form id="asset_form" class="flex flex-col gap-5" method="post" action="<?= htmlspecialchars($HANDLER_URL) ?>">
        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf_token) ?>">
        <input type="hidden" name="return_to" value="<?= htmlspecialchars($RETURN_TO) ?>">
        <?php if ($formMode === 'update' && $editState): ?>
          <input type="hidden" name="client_asset_id" value="<?= (int)$editState['id'] ?>">
        <?php endif; ?>

        <div class="flex flex-col gap-1">
          <label class="kt-form-label font-normal text-mono" for="asset_id">Asset</label>
          <select id="asset_id" name="asset_id" class="kt-input" required>
            <option value="">-- Select Asset --</option>
            <?php foreach ($dropdownAssets as $a): ?>
              <?php $sel = ($prefill['asset_id'] !== '' && (int)$prefill['asset_id'] === (int)$a['id']) ? 'selected' : ''; ?>
              <option value="<?= (int)$a['id'] ?>" data-decimals="<?= (int)$a['decimals'] ?>" <?= $sel ?>>
                <?= htmlspecialchars($a['symbol'] . ' - ' . $a['name']) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="flex flex-col gap-1">
          <label class="kt-form-label font-normal text-mono" for="amount">Amount</label>
          <input id="amount"
                 class="kt-input"
                 type="number"
                 inputmode="decimal"
                 step="<?= htmlspecialchars(decimalsToStepAttr($prefill['decimals'])) ?>"
                 name="amount"
                 placeholder="0.00"
                 value="<?= htmlspecialchars($prefill['amount']) ?>"
                 required />
        </div>

        <div class="flex items-center gap-3">
          <?php if ($formMode === 'update'): ?>
            <button type="submit" name="action" value="update" class="kt-btn kt-btn-primary">
              Update<i class="ki-filled ki-black-right"></i>
            </button>
            <button type="submit" name="action" value="cancel_edit" formnovalidate class="kt-btn kt-btn-light">
              Cancel
            </button>
          <?php else: ?>
            <button type="submit" name="action" value="add" class="kt-btn kt-btn-primary">
              Add<i class="ki-filled ki-black-right"></i>
            </button>
          <?php endif; ?>
        </div>
      </form>

      <!-- Active Assets -->
      <div class="mt-4">
        <h4 class="text-base font-semibold mb-3">Your Active Assets</h4>
        <?php if (count($clientActive) === 0): ?>
          <div class="text-sm text-secondary-foreground">No active assets yet.</div>
        <?php else: ?>
          <ul class="space-y-2" id="assets-active-list">
            <?php foreach ($clientActive as $row): ?>
              <li class="flex items-center justify-between p-3 rounded-md border border-border">
                <div>
                  <div class="font-medium"><?= htmlspecialchars($row['symbol'] . ' - ' . $row['name']) ?></div>
                  <div class="text-sm text-secondary-foreground">Amount: <?= htmlspecialchars(rtrim(rtrim($row['amount'],'0'),'.')) ?></div>
                </div>

                <form method="post" action="<?= htmlspecialchars($HANDLER_URL) ?>">
                  <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf_token) ?>">
                  <input type="hidden" name="return_to" value="<?= htmlspecialchars($RETURN_TO) ?>">
                  <input type="hidden" name="action" value="start_edit">
                  <input type="hidden" name="client_asset_id" value="<?= (int)$row['id'] ?>">
                  <button type="submit" class="kt-btn kt-btn-light">Edit</button>
                </form>
              </li>
            <?php endforeach; ?>
          </ul>
        <?php endif; ?>
      </div>

      <!-- Inactive / Pending -->
      <?php if (count($clientInactive) > 0): ?>
        <div class="mt-6">
          <h4 class="text-base font-semibold mb-3">Your Inactive / Pending Assets</h4>
          <ul class="space-y-2" id="assets-inactive-list">
            <?php foreach ($clientInactive as $row): ?>
              <li class="flex items-center justify-between p-3 rounded-md border border-border">
                <div>
                  <div class="font-medium"><?= htmlspecialchars($row['symbol'] . ' - ' . $row['name']) ?></div>
                  <div class="text-sm text-secondary-foreground">
                    Amount: <?= htmlspecialchars(rtrim(rtrim($row['amount'],'0'),'.')) ?>
                    <span class="ml-2 opacity-70">(Pending admin approval)</span>
                  </div>
                </div>

                <form method="post" action="<?= htmlspecialchars($HANDLER_URL) ?>">
                  <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf_token) ?>">
                  <input type="hidden" name="return_to" value="<?= htmlspecialchars($RETURN_TO) ?>">
                  <input type="hidden" name="action" value="start_edit">
                  <input type="hidden" name="client_asset_id" value="<?= (int)$row['id'] ?>">
                  <button type="submit" class="kt-btn kt-btn-light">Edit</button>
                </form>
              </li>
            <?php endforeach; ?>
          </ul>
        </div>
      <?php endif; ?>
    </div>
  </div>
</section>
