﻿<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }

$messages = [];
$errors   = [];
$fieldErr = [];

require_once __DIR__ . '/../config.php';

try {
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        if (defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER')) {
            $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
            $pdo = new PDO($dsn, DB_USER, defined('DB_PASS') ? DB_PASS : '', [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
        } else {
            throw new RuntimeException('Database connection not available. Ensure config.php sets $pdo or DB_* constants.');
        }
    }
} catch (Throwable $e) {
    $errors[] = 'Database error: ' . htmlspecialchars($e->getMessage());
}

// --- Auth guard ---
$currentUserId = $_SESSION['user_id'] ?? null;
if (!$currentUserId) {
    ?>
    <main id="page-content" class="w-full flex justify-center py-12">
      <div class="kt-card max-w-[600px] w-full">
        <div class="kt-card-content flex flex-col gap-5 p-10">
          <h2 class="text-xl font-semibold text-center">KYC Documents</h2>
          <div class="text-sm p-4 rounded-lg bg-accent/40 text-secondary-foreground text-center">
            You must be logged in to upload your KYC documents.
            <a href="#login" data-page="login" class="text-primary underline ml-1">Go to Login</a>
          </div>
        </div>
      </div>
    </main>
    <?php
    exit;
}

// --- Load active KYC types ---
$activeTypes = [];
if (empty($errors)) {
    try {
        $stmt = $pdo->query("
            SELECT id, type_key,
                   COALESCE(NULLIF(TRIM(description), ''), type_key) AS label
            FROM kyc_document_types
            WHERE is_active = 1
            ORDER BY id ASC
        ");
        $activeTypes = $stmt->fetchAll();
        if (!$activeTypes) {
            $messages[] = 'No active KYC document types are configured yet.';
        }
    } catch (Throwable $e) {
        $errors[] = 'Failed to load KYC document types.';
    }
}

if (!empty($_SESSION['kyc_flash'])) {
    $flash = $_SESSION['kyc_flash'];
    unset($_SESSION['kyc_flash']);
    if (!empty($flash['errors']))   { $errors   = array_merge($errors,   (array)$flash['errors']); }
    if (!empty($flash['messages'])) { $messages = array_merge($messages, (array)$flash['messages']); }
    if (!empty($flash['fieldErr'])) { $fieldErr = (array)$flash['fieldErr']; }
}

// --- CSRF token for KYC flow ---
if (empty($_SESSION['csrf_kyc'])) {
    $_SESSION['csrf_kyc'] = bin2hex(random_bytes(16));
}
$csrfToken = $_SESSION['csrf_kyc'];

$scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? '/'), '/');       // e.g. /novademo/html/demo1 OR .../userdashboard
$appBase  = preg_replace('~(/userdashboard)(?:/.*)?$~', '', $scriptDir); // strip trailing /userdashboard...
if ($appBase === '') { $appBase = '/'; }                                 // just in case

$handlerAction   = rtrim($appBase, '/') . '/userdashboard/kyc_submit.php';
$returnToDefault = rtrim($appBase, '/') . '/index.php#kyc';

function field_error_html($field, $fieldErr) {
    if (empty($fieldErr[$field])) return '';
    $out = '<ul class="text-red-700 mt-1 ml-4 list-disc">';
    foreach ($fieldErr[$field] as $msg) {
        $out .= '<li>' . htmlspecialchars($msg) . '</li>';
    }
    $out .= '</ul>';
    return $out;
}
?>

<main id="page-content" class="w-full flex justify-center py-12">
  <div class="kt-card max-w-[600px] w-full">
    <form method="post"
          enctype="multipart/form-data"
          class="kt-card-content flex flex-col gap-5 p-10"
          id="kyc_form"
          action="<?php echo htmlspecialchars($handlerAction); ?>">
      <div class="text-center">
        <h3 class="text-lg font-medium text-mono">KYC Documents</h3>
        <span class="text-sm text-secondary-foreground">
          Upload a file for each required document (PDF or image, max 5&nbsp;MB each).
        </span>
      </div>

      <?php if (!empty($errors)) : ?>
        <div class="rounded-lg border border-red-300 bg-red-50 p-4 text-red-800">
          <strong>Errors</strong>
          <ul class="list-disc ml-5 mt-2">
            <?php foreach ($errors as $e) : ?>
              <li><?php echo htmlspecialchars($e); ?></li>
            <?php endforeach; ?>
          </ul>
        </div>
      <?php endif; ?>

      <?php if (!empty($messages)) : ?>
        <div class="rounded-lg border border-emerald-300 bg-emerald-50 p-4 text-emerald-800">
          <?php foreach ($messages as $m) : ?>
            <p><?php echo htmlspecialchars($m); ?></p>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>

      <?php if ($currentUserId && $activeTypes) : ?>
        <?php foreach ($activeTypes as $type) : ?>
          <?php $fieldName = 'kyc_' . (int)$type['id']; ?>
          <div class="flex flex-col gap-1">
            <label class="kt-form-label font-normal text-mono" for="<?php echo $fieldName; ?>">
              <?php echo htmlspecialchars($type['label']); ?>
            </label>
            <input
              id="<?php echo $fieldName; ?>"
              class="kt-input"
              type="file"
              name="<?php echo $fieldName; ?>"
              accept=".pdf,image/*"
              required
            />
            <span class="text-xs text-muted-foreground">Accepted: PDF or image (max 5&nbsp;MB). Renamed or fake files will be rejected.</span>
            <?php echo field_error_html($fieldName, $fieldErr); ?>
          </div>
        <?php endforeach; ?>

        <!-- CSRF + return target -->
        <input type="hidden" name="csrf_token"  value="<?php echo htmlspecialchars($csrfToken); ?>">
        <input type="hidden" id="kyc_return_to" name="return_to" value="<?php echo htmlspecialchars($returnToDefault); ?>">

        <!-- Compact, content-wide button -->
        <button type="submit"
                id="kyc_submit_btn"
                name="kyc_submit"
                class="kt-btn kt-btn-primary"
                style="width:max-content !important; display:inline-flex !important; align-self:flex-start !important; gap:.5rem; padding:.6rem 1rem;">
          <span>Submit KYC</span>
          <i class="ki-filled ki-black-right"></i>
        </button>
      <?php elseif ($currentUserId && !$activeTypes) : ?>
        <div class="text-center text-secondary-foreground">No active KYC document types are configured yet.</div>
      <?php endif; ?>
    </form>
  </div>
</main>

<script>
  (function () {
    var rt = document.getElementById('kyc_return_to');
    if (rt) {
      rt.value = window.location.pathname + window.location.search + '#kyc';
    }
  })();
</script>
