<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }

$errors   = [];
$messages = [];
$fieldErr = [];


function compute_app_base(): string {
    $scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? '/'), '/');      // e.g. /novademo/html/demo1/userdashboard
    $appBase  = preg_replace('~(/userdashboard)(?:/.*)?$~', '', $scriptDir);
    if ($appBase === '') { $appBase = '/'; }
    return $appBase;
}
function safe_redirect_target(string $fallback): string {
    $target = $_POST['return_to'] ?? $_SERVER['HTTP_REFERER'] ?? $fallback;
    if (preg_match('~^https?://~i', $target)) { return $fallback; }
    if ($target === '' || $target[0] !== '/') { return $fallback; }
    return $target;
}
function redirect_back() {
    $appBase = compute_app_base();
    $fallback = rtrim($appBase, '/') . '/index.php#kyc';
    $target = safe_redirect_target($fallback);
    header('Location: ' . $target);
    exit;
}

function ini_shorthand_to_bytes(?string $val): int {
    if ($val === null) return 0;
    $val = trim($val);
    if ($val === '') return 0;
    $last = strtolower($val[strlen($val)-1]);
    $num  = (int)$val;
    switch ($last) {
        case 'g': return $num * 1024 * 1024 * 1024;
        case 'm': return $num * 1024 * 1024;
        case 'k': return $num * 1024;
        default:  return (int)$val;
    }
}

// MIME helpers
function detect_mime($filePath): string {
    if (class_exists('finfo')) {
        $fi = new finfo(FILEINFO_MIME_TYPE);
        $m  = $fi->file($filePath);
        if ($m) return $m;
    }
    if (function_exists('mime_content_type')) {
        $m = @mime_content_type($filePath);
        if ($m) return $m;
    }
    return 'application/octet-stream';
}
function is_real_pdf($tmpPath): bool {
    $h = @fopen($tmpPath, 'rb');
    if (!$h) return false;
    $sig = @fread($h, 4);
    @fclose($h);
    return ($sig === "%PDF");
}
function is_real_image($tmpPath): bool {
    $info = @getimagesize($tmpPath);
    return $info !== false;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_POST['kyc_submit'])) {
    $errors[] = 'Invalid request method.';
    $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
    redirect_back();
}

$contentLen = isset($_SERVER['CONTENT_LENGTH']) ? (int)$_SERVER['CONTENT_LENGTH'] : 0;
if (empty($_POST) && empty($_FILES) && $contentLen > 0) {
    $postMax = ini_shorthand_to_bytes(ini_get('post_max_size'));
    if ($postMax > 0 && $contentLen > $postMax) {
        $errors[] = 'Upload too large for server (post_max_size). Please upload smaller files or one by one.';
    } else {
        $errors[] = 'Upload could not be processed. Try smaller files or contact support.';
    }
    $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
    redirect_back();
}

if (empty($_POST['csrf_token']) || empty($_SESSION['csrf_kyc']) || !hash_equals($_SESSION['csrf_kyc'], $_POST['csrf_token'])) {
    $errors[] = 'Session expired or invalid form token. Please try again.';
    $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
    redirect_back();
}

$currentUserId = $_SESSION['user_id'] ?? null;
if (!$currentUserId) {
    $errors[] = 'You must be logged in to submit KYC documents.';
    $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
    redirect_back();
}

require_once __DIR__ . '/../config.php';
try {
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        if (defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER')) {
            $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
            $pdo = new PDO($dsn, DB_USER, defined('DB_PASS') ? DB_PASS : '', [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
        } else {
            throw new RuntimeException('Database connection not available.');
        }
    }
} catch (Throwable $e) {
    $errors[] = 'Database error: ' . htmlspecialchars($e->getMessage());
    $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
    redirect_back();
}

$activeTypes = [];
try {
    $stmt = $pdo->query("
        SELECT id, type_key,
               COALESCE(NULLIF(TRIM(description), ''), type_key) AS label
        FROM kyc_document_types
        WHERE is_active = 1
        ORDER BY id ASC
    ");
    $activeTypes = $stmt->fetchAll();
    if (!$activeTypes) {
        $errors[] = 'No active KYC document types are configured.';
        $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
        redirect_back();
    }
} catch (Throwable $e) {
    $errors[] = 'Failed to load KYC document types.';
    $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
    redirect_back();
}

$maxBytes = 5 * 1024 * 1024; // 5 MB per file

$demoRoot   = realpath(__DIR__ . '/..'); // .../html/demo1
$uploadRoot = $demoRoot . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'kyc' . DIRECTORY_SEPARATOR;
$publicRel  = 'uploads/kyc/';

if (!is_dir($uploadRoot)) { @mkdir($uploadRoot, 0775, true); }
if (!is_dir($uploadRoot) || !is_writable($uploadRoot)) {
    $errors[] = 'Upload directory is missing or not writable: ' . htmlspecialchars($uploadRoot);
    $_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
    redirect_back();
}

$anySuccess     = false;
$hadFieldErrors = false;

$insert = $pdo->prepare("
    INSERT INTO kyc_documents
        (user_id, type_id, file_path, status, submitted_at, reviewed_at, reviewed_by_user_id, reject_reason)
    VALUES
        (:user_id, :type_id, :file_path, :status, NOW(), NULL, NULL, NULL)
");

foreach ($activeTypes as $type) {
    $tid   = (int)$type['id'];
    $label = (string)$type['label'];
    $field = 'kyc_' . $tid;

    if (empty($_FILES[$field]) || ($_FILES[$field]['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) {
        $fieldErr[$field][] = $label . '  no file selected.';
        $hadFieldErrors = true;
        continue;
    }

    $file = $_FILES[$field];

    if ($file['error'] !== UPLOAD_ERR_OK) {
        $fieldErr[$field][] = $label . '  upload error (code ' . (int)$file['error'] . ').';
        $hadFieldErrors = true;
        continue;
    }

    if ($file['size'] > $maxBytes) {
        $fieldErr[$field][] = $label . '  file exceeds 5 MB.';
        $hadFieldErrors = true;
        continue;
    }

    $mime   = detect_mime($file['tmp_name']) ?: 'application/octet-stream';
    $isPdf  = ($mime === 'application/pdf') && is_real_pdf($file['tmp_name']);
    $isImg  = (strpos($mime, 'image/') === 0) && is_real_image($file['tmp_name']);

    if (!($isPdf || $isImg)) {
        $fieldErr[$field][] = $label . '  only real images or real PDF files are allowed.';
        $hadFieldErrors = true;
        continue;
    }

    if ($isPdf) {
        $ext = '.pdf';
    } else {
        $map = [
            IMAGETYPE_JPEG    => '.jpg',
            IMAGETYPE_PNG     => '.png',
            IMAGETYPE_GIF     => '.gif',
            IMAGETYPE_WEBP    => '.webp',
            IMAGETYPE_BMP     => '.bmp',
            IMAGETYPE_TIFF_II => '.tiff',
            IMAGETYPE_TIFF_MM => '.tiff',
        ];
        $imgInfo = @getimagesize($file['tmp_name']);
        $ext = isset($imgInfo[2], $map[$imgInfo[2]]) ? $map[$imgInfo[2]] : ('.' . strtolower(pathinfo($file['name'], PATHINFO_EXTENSION)));
        if (!preg_match('/^\.(jpg|jpeg|png|gif|webp|bmp|tiff)$/i', $ext)) {
            $ext = '.jpg';
        }
    }

    $safeLabel = preg_replace('/[^a-zA-Z0-9\-_]/', '_', $label);
    $filename = sprintf(
        'u%s_type%s_%s_%s%s',
        (int)$currentUserId,
        $tid,
        $safeLabel,
        date('Ymd_His') . '_' . bin2hex(random_bytes(4)),
        $ext
    );

    $targetAbs = $uploadRoot . $filename;
    $targetRel = $publicRel . $filename;

    if (!move_uploaded_file($file['tmp_name'], $targetAbs)) {
        $fieldErr[$field][] = $label . '  failed to save the uploaded file.';
        $hadFieldErrors = true;
        continue;
    }

    try {
        $insert->execute([
            ':user_id'   => (int)$currentUserId,
            ':type_id'   => $tid,
            ':file_path' => $targetRel,
            ':status'    => 'submitted',
        ]);
        $anySuccess = true;
        $messages[] = $label . ' - uploaded successfully.';
    } catch (Throwable $e) {
        @unlink($targetAbs);
        $fieldErr[$field][] = $label . '  database error while saving record.';
        $hadFieldErrors = true;
    }
}

if ($hadFieldErrors) {
    $errors[] = 'Some files have errors. Please check the messages below each field.';
}
if (!$anySuccess && !$hadFieldErrors) {
    $errors[] = 'No documents were uploaded. Please try again.';
}

$_SESSION['kyc_flash'] = compact('errors','messages','fieldErr');
redirect_back();
